/*************************************************************************
 * The contents of this file are subject to the MYRICOM MX AND GM-2      *
 * MAPPING SOFTWARE AND DOCUMENTATION LICENSE (the "License"); User may  *
 * not use this file except in compliance with the License.  The full    *
 * text of the License can found in mapper directory in LICENSE.TXT      *
 *                                                                       *
 * Software distributed under the License is distributed on an "AS IS"   *
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See  *
 * the License for the specific language governing rights and            *
 * limitations under the License.                                        *
 *                                                                       *
 * Copyright 2003 - 2004 by Myricom, Inc.  All rights reserved.          *
 *************************************************************************/

#include <stdarg.h>

#include "lx.h"
#include "lx_map_file.h"
#include "lx_routing.h"
#include "lx_route_file.h"

lx_t lx;

static void usage (char*s)
{
  fprintf (stderr, "usage: %s <map file>\n", s);
  fprintf (stderr, "options:\n");
  fprintf (stderr, "  --mac-address=<mac address> (to compute routes for just 1 host)\n");
  fprintf (stderr, "  --one (to compute routes for just 1st host in file)\n");
  fprintf (stderr, "  --num-routes=<num routes per port> (default is 1)\n");
  fprintf (stderr, "  --not-shortest-path\n");
  fprintf (stderr, "  --forget (local computation)\n");
  fprintf (stderr, "  --bfs (breadth first search)\n");
  fprintf (stderr, "  --seek-size=<number>\n");
  
  exit (0);
}

int main (int argc, char*argv [])
{
  unsigned char mac_address [6];
  char s [100];
  lx_node_t*me, *first;
  lx_map_t*m;
  FILE*fp;
  int a [6];
  int has_address = 0;
  int num_spines, i, j, k, t;
  int num_routes = 1;
  int num_ports = 2;
  int show_progress = 1;
  int shortest = 1;
  int bfs = 0;
  int one = 0;
  int forget = 0;
  int seed;
  int seek_size = 1;
  
  mi_verbose_flag = 0;
  
  if (argc < 2)
    usage (argv [0]);
  
  if (!(fp = fopen (argv [1], "r")))
  {
    fprintf (stderr, "couldn't open %s\n", argv [1]);
    usage (argv [0]);
  }
  
  for (i = 2; i < argc; i++)
  {
    if (strstr (argv [i], "--num-routes="))
    { 
      if (sscanf (argv [i], "--num-routes=%d", &num_routes) != 1 || num_routes <= 0)
	usage (argv [0]);
    }
    else if (strstr (argv [i], "--seek-size="))
    { 
      if (sscanf (argv [i], "--seek-size=%d", &seek_size) != 1 || seek_size < 0)
	usage (argv [0]);
    }
    else if (strstr (argv [i], "--mac-address="))
    {
      if (sscanf (argv [i], "--mac-address=%x:%x:%x:%x:%x:%x", &a [0], &a [1], &a [2], &a [3], &a [4], &a [5]) != 6)
	usage (argv [0]);

      for (j = 0; j < 6; j++)
	mac_address [j] = a [j];

      has_address = 1;
    }
    else if (strstr (argv [i], "--not-shortest"))
      shortest = 0;
    else if (strstr (argv [i], "--bfs"))
      bfs = 1;
    else if (strstr (argv [i], "--forget"))
      forget = 1;
    else if (strstr (argv [i], "--one"))
      one = 1;
    else usage (argv [0]);
  }
  
  lx_init (&lx);
  
  if (!lx_map_file_read (fp, &lx.maps [0], &first))
  {
    fprintf (stderr, "lx_map_file_read () failed\n");
    return 0;
  }

  m = lx.maps;
  fprintf (stderr, "%d hosts and %d xbars\n", m->num_hosts, m->num_xbars);

  if (has_address && !(me = lx_seen (m, mac_address)))
  {
    fprintf (stderr, lx_mac_format " not found in %s\n", lx_mac_args (mac_address), argv [1]);
    return 0;
  }  
  
  if (one)
  {
    has_address = 1;
    me = m->host_array [0];
  }
  
  fclose (fp);
  fp = stdout;
  
  lx_routing_forget (m);
  
  *s = 0;
  
  for (i = 0; i < m->num_hosts && (!has_address || !i); i++)
  {
    if (show_progress)
    {
      for (j = 0; j < (int) strlen (s); j++)
	fprintf (stderr, "\b");
      sprintf (s, "%2.1f%% (%d)", (double) i * 100 / m->num_hosts, i);
      fprintf (stderr, "%s",s);
      fflush (stderr);
    }
    
    if (!has_address)
      me = m->host_array [i];
    m->root = me;    

    if (!lx_route_file_write_header (fp, m, me, num_ports, num_routes))
      return 0;

    memcpy ((char*) &seed, lx_host_c (me)->mac_address + 2, 4);
    lx_seed (seed);
    
    if (forget)
      lx_routing_forget (m);

    for (k = 0; k < num_ports; k++)
    {      
      lx_number_xbars_bfs (m, k);

      if (!bfs && !(m->max_level = lx_routing_number_xbars_clos (m, &num_spines))
	  && m->num_xbars)
	return 0;

      for (t = 0; t < num_routes; t++)
      {  
	if ((bfs && !lx_routing_shortest_path (m, k, me, !shortest)) ||
	    (!bfs && !lx_routing_careful_routes (m, k, me, seek_size, LX_ROUTING_LEAST, shortest)) ||
	    !lx_route_file_write_routes (fp, m, me, k, t))
	  return 0;
      }
    }
  }
  
  if (show_progress) 
    fprintf (stderr, "\n");
  
  return 1;
}
